;****************************************************************
;*  gbemu_rw.s
;*			Emulation of the Gameboy CPU
;*			(Reading and writing memory)
;*
;*  2000-04-24  Bodo Wenzel  Creation
;*  2000-05-06  Bodo Wenzel  External RAM pageable
;*  2000-06-06  Bodo Wenzel  New memory management
;*  2000-08-05  Bodo Wenzel  Some improvements
;*  2000-10-02  Bodo Wenzel  Interrupts
;*  2001-01-31  Bodo Wenzel  Support of more cartridges
;****************************************************************
;
;  (c)2000 Bodo Wenzel
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
;****************************************************************

;================================================================
; All memory accesses are generated with:
;	d0.b	byte to be written
;	d1.b	lower address byte (mem_rd: must be preserved)
;	-	higher address byte (is implicit)
;	d2.b	(mem_rd: must be preserved)
;	a2	68000 base memory address (used after mem_rd)
;	a3	68000 memory address (used after mem_rd,
;		  also as "mem_lea")
;	a6	68000 return address

;================================================================
; Macros for generating macros...

macro	rd003f	num		;ROM bank 0
macro	rd&num
	local	rd
	move.l	BASE_ROM_0(a4),a2
	move.b	d1,rd+3-gbemu(a4)
rd:	lea	DONT_OPT_LO+(num << 8)(a2),a3
	jmp	(a6)
	endm
	endm

macro	rd407f	num		;ROM bank 1
macro	rd&num
	local	rd
	move.l	BASE_ROM_1(a4),a2
	move.b	d1,rd+3-gbemu(a4)
rd:	lea	DONT_OPT_LO+(num << 8)(a2),a3
	jmp	(a6)
	endm
	endm

macro	wr001f	num		;RAM bank enable (MCB1, MCB2)
macro	wr&num
	jmp	V_RAM_ENABLE(a4)
	endm
	endm

macro	wr202f	num		;ROM bank select
macro	wr&num
	jmp	V_ROML_SELECT(a4)
	endm
	endm

macro	wr303f	num		;ROM bank select
macro	wr&num
	jmp	V_ROMH_SELECT(a4)
	endm
	endm

macro	wr405f	num		;RAM bank select
macro	wr&num
	jmp	V_RAM_SELECT(a4)
	endm
	endm

macro	wr607f	num		;ROM/RAM mode select (MCB1)
macro	wr&num
	jmp	V_MODE_SELECT(a4)
	endm
	endm

;----------------------------------------------------------------

macro	rd809f	num		;RAM video read
macro	rd&num
	local	rd
	move.l	BASE_RAM_VID(a4),a2
	move.b	d1,rd+3-gbemu(a4)
rd:	lea	DONT_OPT_LO+(num << 8)(a2),a3
	jmp	(a6)
	endm
	endm

macro	wr8087	num		;RAM tile tab $8000..$87ff write
macro	wr&num
	move.l	BASE_RAM_VID(a4),a2
	and.w	#$ff,d1
	or.w	#num << 8,d1
	jmp	V_RAM_WR8087(a4)
	endm
	endm

macro	wr888f	num		;RAM tile tab $8800..$8fff write
macro	wr&num
	move.l	BASE_RAM_VID(a4),a2
	and.w	#$ff,d1
	or.w	#num << 8,d1
	jmp	V_RAM_WR888F(a4)
	endm
	endm

macro	wr9097	num		;RAM tile tab $9000..$97ff write
macro	wr&num
	move.l	BASE_RAM_VID(a4),a2
	and.w	#$ff,d1
	or.w	#num << 8,d1
	jmp	V_RAM_WR9097(a4)
	endm
	endm

macro	wr989b	num		;RAM screen $9800..$9bff write
macro	wr&num
	move.l	BASE_RAM_VID(a4),a2
	and.w	#$ff,d1
	or.w	#num << 8,d1
	jmp	V_RAM_WR989B(a4)
	endm
	endm

macro	wr9c9f	num		;RAM screen $9c00..$9fff write
macro	wr&num
	move.l	BASE_RAM_VID(a4),a2
	and.w	#$ff,d1
	or.w	#num << 8,d1
	jmp	V_RAM_WR9C9F(a4)
	endm
	endm

;----------------------------------------------------------------

macro	rda0bf	num		;RAM external read
macro	rd&num
	and.w	#$ff,d1
	or.w	#num << 8,d1
	jmp	V_RAM_RD_EXT(a4)
	endm
	endm

macro	wra0bf	num		;RAM external write
macro	wr&num
	and.w	#$ff,d1
	or.w	#num << 8,d1
	jmp	V_RAM_WR_EXT(a4)
	endm
	endm

macro	rdc0df	num		;RAM internal read
macro	rd&num
	local	rd
	move.l	BASE_RAM_INT(a4),a2
	move.b	d1,rd+3-gbemu(a4)
rd:	lea	DONT_OPT_LO+(num << 8)(a2),a3
	jmp	(a6)
	endm
	endm

macro	wrc0df	num		;RAM internal write
macro	wr&num
	local	wr
	move.l	BASE_RAM_INT(a4),a2
	move.b	d1,wr+3-gbemu(a4)
wr:	move.b	d0,DONT_OPT_LO+(num << 8)(a2)
	jmp	(a6)
	endm
	endm

macro	rde0fd	num		;RAM internal read (mirror)
macro	rd&num
	local	rd
	move.l	BASE_RAM_INT(a4),a2
	move.b	d1,rd+3-gbemu(a4)
rd:	lea	DONT_OPT_LO+((num-$20) << 8)(a2),a3
	jmp	(a6)
	endm
	endm

macro	wre0fd	num		;RAM internal write (mirror)
macro	wr&num
	local	wr
	move.l	BASE_RAM_INT(a4),a2
	move.b	d1,wr+3-gbemu(a4)
wr:	move.b	d0,DONT_OPT_LO+((num-$20) << 8)(a2)
	jmp	(a6)
	endm
	endm

;================================================================

i	set	$00		;start with page $00

	rept	$20
	rd003f	$(i)		;1st half of ROM bank 0
	wr001f	$(i)		;RAM bank enable
i	set	i+1
	endm

	rept	$10
	rd003f	$(i)		;3rd quarter of ROM bank 0
	wr202f	$(i)		;ROM bank select
i	set	i+1
	endm

	rept	$10
	rd003f	$(i)		;4th quarter of ROM bank 0
	wr303f	$(i)		;ROM bank select
i	set	i+1
	endm

	rept	$20
	rd407f	$(i)		;1st half of ROM bank 1
	wr405f	$(i)		;RAM bank select
i	set	i+1
	endm

	rept	$20
	rd407f	$(i)		;2nd half of ROM bank 1
	wr607f	$(i)		;ROM/RAM mode select
i	set	i+1
	endm

	rept	$08
	rd809f	$(i)		;RAM tile tab $8000..$87ff
	wr8087	$(i)
i	set	i+1
	endm

	rept	$08
	rd809f	$(i)		;RAM tile tab $8800..$8fff
	wr888f	$(i)
i	set	i+1
	endm

	rept	$08
	rd809f	$(i)		;RAM tile tab $9000..$97ff
	wr9097	$(i)
i	set	i+1
	endm

	rept	$04
	rd809f	$(i)		;RAM screen $9800..$9bff
	wr989b	$(i)
i	set	i+1
	endm

	rept	$04
	rd809f	$(i)		;RAM screen $9c00..$9fff
	wr9c9f	$(i)
i	set	i+1
	endm

	rept	$20
	rda0bf	$(i)		;RAM external
	wra0bf	$(i)
i	set	i+1
	endm

	rept	$20
	rdc0df	$(i)		;RAM internal
	wrc0df	$(i)
i	set	i+1
	endm

	rept	$1e
	rde0fd	$(i)		;RAM internal (mirror)
	wre0fd	$(i)
i	set	i+1
	endm

macro	rd$fe			;RAM OAM read
	local	rd
	move.l	BASE_RAM_OAM(a4),a2
	move.b	d1,rd+3-gbemu(a4)
rd:	lea	DONT_OPT_LO+($fe << 8)(a2),a3
	jmp	(a6)
	endm

macro	wr$fe			;RAM OAM write
	move.l	BASE_RAM_OAM(a4),a2
	and.w	#$ff,d1
	or.w	#$fe << 8,d1
	jmp	V_RAM_WR_OAM(a4)
	endm

macro	rd$ff			;input
	local	jump
	move.b	d1,jump+2-gbemu(a4)
jump:	jmp	OFFSET_IN+DONT_OPT_HI(a4)
	endm

macro	wr$ff			;output
	local	jump
	move.b	d1,jump+2-gbemu(a4)
jump:	jmp	OFFSET_OUT+DONT_OPT_HI(a4)
	endm

;================================================================
;	end of include
