;****************************************************************
;*  gbemu_op.s
;*			Emulation of the Gameboy CPU
;*			("Normal" opcodes)
;*
;*  2000-02-23  Bodo Wenzel  Creation
;*  2000-06-18  Bodo Wenzel  New memory management
;*  2000-08-23  Bodo Wenzel  Some improvements
;*  2000-09-27  Bodo Wenzel  DAA implemented
;*  2000-10-03  Bodo Wenzel  Interrupts
;*  2000-12-02  Bodo Wenzel  HALT with DI support
;*  2001-02-01  Bodo Wenzel  Correction at debug modes
;*  2001-02-05  Bodo Wenzel  New stack management
;****************************************************************
;
;  (c)2000 Bodo Wenzel
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
;****************************************************************

macro	op$0			;nop
	exec_pc
	endm

macro	op$1			;ld bc,##
	move.b	(a0)+,d0
	move.b	(a0)+,d5
	swap	d5
	move.b	d0,d5
	next_pc
	endm

macro	op$2			;ld (bc),a
	move.b	d7,d0
	move.b	d5,d1
	move.l	d5,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$3			;inc bc
	ext.w	d5
	addq.l	#1,d5
	next_pc
	endm

macro	op$4			;inc b
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	swap	d5
	addq.b	#1,d5
	beq.s	z
	bchg	d0,d6
z:	swap	d5
	next_pc
	endm

macro	op$5			;dec b
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	swap	d5
	subq.b	#1,d5
	beq.s	z
	bchg	d0,d6
z:	swap	d5
	next_pc
	endm

macro	op$6			;ld b,#
	swap	d5
	move.b	(a0)+,d5
	swap	d5
	next_pc
	endm

macro	op$7			;rlca
	local	c
	moveq	#FLAG_C,d0
	bset	d0,d6
	rol.b	#1,d7
	bcs.s	c
	bchg	d0,d6
c:	next_pc
	endm

macro	op$8			;ld (##),sp
	local	go_on,quit
	move.b	(a0)+,d1
	move.b	(a0)+,d2
	if	DEBUG & DEBUG_NO_RW
	bmi.s	go_on
	return	ERR_DEBUG
	else
	bpl.s	quit
	endif
go_on:	mem_rd	d2		;used as "get mem addr"
	move.b	CPU_STATE_SP+1(a4),(a3)+
	move.b	CPU_STATE_SP+0(a4),(a3)
quit:	next_pc
	endm

macro	op$9			;add hl,bc
	local	c
	moveq	#FLAG_C,d0
	bset	d0,d6
	add.b	d5,d3
	swap	d3
	swap	d5
	addx.b	d5,d3
	bcs.s	c
	bchg	d0,d6
c:	swap	d3
	swap	d5
	next_pc
	endm

macro	op$a			;ld a,(bc)
	move.b	d5,d1
	move.l	d5,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d7
	next_pc
	endm

macro	op$b			;dec bc
	ext.w	d5
	subq.l	#1,d5
	next_pc
	endm

macro	op$c			;inc c
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	addq.b	#1,d5
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$d			;dec c
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	subq.b	#1,d5
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$e			;ld c,#
	move.b	(a0)+,d5
	next_pc
	endm

macro	op$f			;rrca
	local	c
	moveq	#FLAG_C,d0
	bset	d0,d6
	ror.b	#1,d7
	bcs.s	c
	bchg	d0,d6
c:	next_pc
	endm

macro	op$10			;stop
	local	go_on
	move.b	(a0)+,d0
	beq.s	go_on
	return	OPCODE_STOP
go_on:
	jsr	V_OPCODE_STOP(a4)
	next_pc
	endm

macro	op$11			;ld de,##
	move.b	(a0)+,d0
	move.b	(a0)+,d4
	swap	d4
	move.b	d0,d4
	next_pc
	endm

macro	op$12			;ld (de),a
	move.b	d7,d0
	move.b	d4,d1
	move.l	d4,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$13			;inc de
	ext.w	d4
	addq.l	#1,d4
	next_pc
	endm

macro	op$14			;inc d
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	swap	d4
	addq.b	#1,d4
	beq.s	z
	bchg	d0,d6
z:	swap	d4
	next_pc
	endm

macro	op$15			;dec d
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	swap	d4
	subq.b	#1,d4
	beq.s	z
	bchg	d0,d6
z:	swap	d4
	next_pc
	endm

macro	op$16			;ld d,#
	swap	d4
	move.b	(a0)+,d4
	swap	d4
	next_pc
	endm

macro	op$17			;rla
	local	c
	swap	d6
	move	d6,ccr
	swap	d6
	moveq	#FLAG_C,d0
	bset	d0,d6
	roxl.b	#1,d7
	bcs.s	c
	bchg	d0,d6
c:	next_pc
	endm

macro	op$18			;jr #
	move.b	(a0)+,d0
	ext.w	d0
	add.w	d0,a0
	next_pc
	endm

macro	op$19			;add hl,de
	local	c
	moveq	#FLAG_C,d0
	bset	d0,d6
	add.b	d4,d3
	swap	d3
	swap	d4
	addx.b	d4,d3
	bcs.s	c
	bchg	d0,d6
c:	swap	d3
	swap	d4
	next_pc
	endm

macro	op$1a			;ld a,(de)
	move.b	d4,d1
	move.l	d4,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d7
	next_pc
	endm

macro	op$1b			;dec de
	ext.w	d4
	subq.l	#1,d4
	next_pc
	endm

macro	op$1c			;inc e
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	addq.b	#1,d4
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$1d			;dec e
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	subq.b	#1,d4
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$1e			;ld e,#
	move.b	(a0)+,d4
	next_pc
	endm

macro	op$1f			;rra
	local	c
	swap	d6
	move	d6,ccr
	swap	d6
	moveq	#FLAG_C,d0
	bset	d0,d6
	roxr.b	#1,d7
	bcs.s	c
	bchg	d0,d6
c:	next_pc
	endm

macro	op$20			;jr nz,#
	local	quit
	move.b	(a0)+,d1
	btst	#FLAG_Z,d6
	bne.s	quit
	ext.w	d1
	add.w	d1,a0
quit:	next_pc
	endm

macro	op$21			;ld hl,##
	move.b	(a0)+,d0
	move.b	(a0)+,d3
	swap	d3
	move.b	d0,d3
	next_pc
	endm

macro	op$22			;ldi (hl),a
	move.b	d7,d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	ext.w	d3
	addq.l	#1,d3
	mem_wr	d2
	next_pc
	endm

macro	op$23			;inc hl
	ext.w	d3
	addq.l	#1,d3
	next_pc
	endm

macro	op$24			;inc h
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	swap	d3
	addq.b	#1,d3
	beq.s	z
	bchg	d0,d6
z:	swap	d3
	next_pc
	endm

macro	op$25			;dec h
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	swap	d3
	subq.b	#1,d3
	beq.s	z
	bchg	d0,d6
z:	swap	d3
	next_pc
	endm

macro	op$26			;ld h,#
	swap	d3
	move.b	(a0)+,d3
	swap	d3
	next_pc
	endm

macro	op$27			;daa
	local	nc,a,s
	move.b	d7,d0
	move.b	CPU_STATE_A_OLD(a4),d7
	btst	#FLAG_C,d6
	beq.s	nc
	sub.b	d7,d0
	bcc.s	s
a:	swap	d6
	move	#MASK_Z,ccr
	abcd	d0,d7
	move	sr,d6
	swap	d6
	next_pc
nc:	sub.b	d7,d0
	bcc.s	a
s:	swap	d6
	neg.b	d0
	move	#MASK_Z,ccr
	sbcd	d0,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$28			;jr z,#
	local	quit
	move.b	(a0)+,d1
	btst	#FLAG_Z,d6
	beq.s	quit
	ext.w	d1
	add.w	d1,a0
quit:	next_pc
	endm

macro	op$29			;add hl,hl
	local	c
	moveq	#FLAG_C,d0
	bset	d0,d6
	add.b	d3,d3
	swap	d3
	addx.b	d3,d3
	bcs.s	c
	bchg	d0,d6
c:	swap	d3
	next_pc
	endm

macro	op$2a			;ldi a,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	ext.w	d3
	addq.l	#1,d3
	mem_rd	d2
	move.b	(a3),d7
	next_pc
	endm

macro	op$2b			;dec hl
	ext.w	d3
	subq.l	#1,d3
	next_pc
	endm

macro	op$2c			;inc l
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	addq.b	#1,d3
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$2d			;dec l
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	subq.b	#1,d3
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$2e			;ld l,#
	move.b	(a0)+,d3
	next_pc
	endm

macro	op$2f			;cpl
	not.b	d7
	next_pc
	endm

macro	op$30			;jr nc,#
	local	quit
	move.b	(a0)+,d1
	btst	#FLAG_C,d6
	bne.s	quit
	ext.w	d1
	add.w	d1,a0
quit:	next_pc
	endm

macro	op$31			;ld sp,##
	move.b	(a0)+,d1
	move.b	d1,CPU_STATE_SP+1(a4)
	move.b	(a0)+,d2
	move.b	d2,CPU_STATE_SP+0(a4)
	mem_rd	d2
	move.l	a2,CPU_STATE_SP_B(a4)
	move.l	a3,a1
	next_pc
	endm

macro	op$32			;ldd (hl),a
	move.b	d7,d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	ext.w	d3
	subq.l	#1,d3
	mem_wr	d2
	next_pc
	endm

macro	op$33			;inc sp
	local	go_on
	addq.w	#1,a1
	addq.b	#1,CPU_STATE_SP+1(a4)
	bcc.s	go_on
	move.b	CPU_STATE_SP+1(a4),d1
	addq.b	#1,CPU_STATE_SP+0(a4)
	mem_rd	CPU_STATE_SP+0(a4)
	move.l	a2,CPU_STATE_SP_B(a4)
	move.l	a3,a1
go_on:	next_pc
	endm

macro	op$34			;inc (hl)
	local	z
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d0
	bset	#FLAG_Z,d6
	addq.b	#1,d0
	beq.s	z
	bchg	#FLAG_Z,d6
z:	mem_wr	d2
	next_pc
	endm

macro	op$35			;dec (hl)
	local	z
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d0
	bset	#FLAG_Z,d6
	subq.b	#1,d0
	beq.s	z
	bchg	#FLAG_Z,d6
z:	mem_wr	d2
	next_pc
	endm

macro	op$36			;ld (hl),#
	move.b	(a0)+,d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$37			;scf
	bset	#FLAG_C,d6
	next_pc
	endm

macro	op$38			;jr c,#
	local	quit
	move.b	(a0)+,d1
	btst	#FLAG_C,d6
	beq.s	quit
	ext.w	d1
	add.w	d1,a0
quit:	next_pc
	endm

macro	op$39			;add hl,sp
	local	c
	moveq	#FLAG_C,d0
	bset	d0,d6
	add.b	CPU_STATE_SP+1(a4),d3
	swap	d3
	move.b	CPU_STATE_SP+0(a4),d1
	addx.b	d1,d3
	bcs.s	c
	bchg	d0,d6
c:	swap	d3
	next_pc
	endm

macro	op$3a			;ldd a,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	ext.w	d3
	subq.l	#1,d3
	mem_rd	d2
	move.b	(a3),d7
	next_pc
	endm

macro	op$3b			;dec sp
	local	go_on
	subq.w	#1,a1
	subq.b	#1,CPU_STATE_SP+1(a4)
	bcc.s	go_on
	move.b	CPU_STATE_SP+1(a4),d1
	subq.b	#1,CPU_STATE_SP+0(a4)
	mem_rd	CPU_STATE_SP+0(a4)
	move.l	a2,CPU_STATE_SP_B(a4)
	move.l	a3,a1
go_on:	next_pc
	endm

macro	op$3c			;inc a
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	move.b	d7,CPU_STATE_A_OLD(a4)
	addq.b	#1,d7
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$3d			;dec a
	local	z
	moveq	#FLAG_Z,d0
	bset	d0,d6
	move.b	d7,CPU_STATE_A_OLD(a4)
	subq.b	#1,d7
	beq.s	z
	bchg	d0,d6
z:	next_pc
	endm

macro	op$3e			;ld a,#
	move.b	(a0)+,d7
	next_pc
	endm

macro	op$3f			;ccf
	bchg	#FLAG_C,d6
	next_pc
	endm

macro	op$40			;ld b,b
	next_pc
	endm

macro	op$41			;ld b,c
	move.b	d5,d0
	swap	d5
	move.b	d0,d5
	next_pc
	endm

macro	op$42			;ld b,d
	move.l	d4,d0
	move.b	d5,d0
	move.l	d0,d5
	next_pc
	endm

macro	op$43			;ld b,e
	swap	d5
	move.b	d4,d5
	swap	d5
	next_pc
	endm

macro	op$44			;ld b,h
	move.l	d3,d0
	move.b	d5,d0
	move.l	d0,d5
	next_pc
	endm

macro	op$45			;ld b,l
	swap	d5
	move.b	d3,d5
	swap	d5
	next_pc
	endm

macro	op$46			;ld b,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	swap	d5
	move.b	(a3),d5
	swap	d5
	next_pc
	endm

macro	op$47			;ld b,a
	swap	d5
	move.b	d7,d5
	swap	d5
	next_pc
	endm

macro	op$48			;ld c,b
	move.l	d5,d0
	swap	d0
	move.b	d0,d5
	next_pc
	endm

macro	op$49			;ld c,c
	next_pc
	endm

macro	op$4a			;ld c,d
	swap	d4
	move.b	d4,d5
	swap	d4
	next_pc
	endm

macro	op$4b			;ld c,e
	move.b	d4,d5
	next_pc
	endm

macro	op$4c			;ld c,h
	swap	d3
	move.b	d3,d5
	swap	d3
	next_pc
	endm

macro	op$4d			;ld c,l
	move.b	d3,d5
	next_pc
	endm

macro	op$4e			;ld c,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d5
	next_pc
	endm

macro	op$4f			;ld c,a
	move.b	d7,d5
	next_pc
	endm

macro	op$50			;ld d,b
	move.l	d5,d0
	move.b	d4,d0
	move.l	d0,d4
	next_pc
	endm

macro	op$51			;ld d,c
	swap	d4
	move.b	d5,d4
	swap	d4
	next_pc
	endm

macro	op$52			;ld d,d
	next_pc
	endm

macro	op$53			;ld d,e
	move.b	d4,d0
	swap	d4
	move.b	d0,d4
	next_pc
	endm

macro	op$54			;ld d,h
	move.l	d3,d0
	move.b	d4,d0
	move.l	d0,d4
	next_pc
	endm

macro	op$55			;ld d,l
	swap	d4
	move.b	d3,d4
	swap	d4
	next_pc
	endm

macro	op$56			;ld d,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	swap	d4
	move.b	(a3),d4
	swap	d4
	next_pc
	endm

macro	op$57			;ld d,a
	swap	d4
	move.b	d7,d4
	swap	d4
	next_pc
	endm

macro	op$58			;ld e,b
	swap	d5
	move.b	d5,d4
	swap	d5
	next_pc
	endm

macro	op$59			;ld e,c
	move.b	d5,d4
	next_pc
	endm

macro	op$5a			;ld e,d
	move.l	d4,d0
	swap	d0
	move.b	d0,d4
	next_pc
	endm

macro	op$5b			;ld e,e
	next_pc
	endm

macro	op$5c			;ld e,h
	swap	d3
	move.b	d3,d4
	swap	d3
	next_pc
	endm

macro	op$5d			;ld e,l
	move.b	d3,d4
	next_pc
	endm

macro	op$5e			;ld e,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d4
	next_pc
	endm

macro	op$5f			;ld e,a
	move.b	d7,d4
	next_pc
	endm

macro	op$60			;ld h,b
	move.l	d5,d0
	move.b	d3,d0
	move.l	d0,d3
	next_pc
	endm

macro	op$61			;ld h,c
	swap	d3
	move.b	d5,d3
	swap	d3
	next_pc
	endm

macro	op$62			;ld h,d
	move.l	d4,d0
	move.b	d3,d0
	move.l	d0,d3
	next_pc
	endm

macro	op$63			;ld h,e
	swap	d3
	move.b	d4,d3
	swap	d3
	next_pc
	endm

macro	op$64			;ld h,h
	next_pc
	endm

macro	op$65			;ld h,l
	move.b	d3,d0
	swap	d3
	move.b	d0,d3
	next_pc
	endm

macro	op$66			;ld h,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	swap	d3
	move.b	(a3),d3
	swap	d3
	next_pc
	endm

macro	op$67			;ld h,a
	swap	d3
	move.b	d7,d3
	swap	d3
	next_pc
	endm

macro	op$68			;ld l,b
	swap	d5
	move.b	d5,d3
	swap	d5
	next_pc
	endm

macro	op$69			;ld l,c
	move.b	d5,d3
	next_pc
	endm

macro	op$6a			;ld l,d
	swap	d4
	move.b	d4,d3
	swap	d4
	next_pc
	endm

macro	op$6b			;ld l,e
	move.b	d4,d3
	next_pc
	endm

macro	op$6c			;ld l,h
	move.l	d3,d0
	swap	d0
	move.b	d0,d3
	next_pc
	endm

macro	op$6d			;ld l,l
	next_pc
	endm

macro	op$6e			;ld l,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d3
	next_pc
	endm

macro	op$6f			;ld l,a
	move.b	d7,d3
	next_pc
	endm

macro	op$70			;ld (hl),b
	move.l	d5,d0
	swap	d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$71			;ld (hl),c
	move.b	d5,d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$72			;ld (hl),d
	move.l	d4,d0
	swap	d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$73			;ld (hl),e
	move.b	d4,d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$74			;ld (hl),h
	move.l	d3,d0
	swap	d0
	move.b	d3,d1
	move.b	d0,d2
	mem_wr	d2
	next_pc
	endm

macro	op$75			;ld (hl),l
	move.b	d3,d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$76			;halt, no GB quirk emulation
	local	self,jump
	move.b	INT_IF(a4),d0
	not.b	d0
	move.b	d0,op$76if-gbemu(a4)
self:	jsr	V_LCD_LINE(a4)
	move.b	INT_IF(a4),d0
	and.b	op$76if-gbemu(a4),d0
	if	DEBUG & DEBUG_STEP
	bne.s	jump
	subq.l	#1,a0
jump:	return	SINGLE_STEP
	else
	beq.s	self
	move.b	(a0)+,jump+2-gbemu(a4)
jump:	jmp	OFFSET_OP+DONT_OPT_HI(a4)
	endif
op$76if:
	dc.b	0
	endm

macro	op$77			;ld (hl),a
	move.b	d7,d0
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_wr	d2
	next_pc
	endm

macro	op$78			;ld a,b
	swap	d5
	move.b	d5,d7
	swap	d5
	next_pc
	endm

macro	op$79			;ld a,c
	move.b	d5,d7
	next_pc
	endm

macro	op$7a			;ld a,d
	swap	d4
	move.b	d4,d7
	swap	d4
	next_pc
	endm

macro	op$7b			;ld a,e
	move.b	d4,d7
	next_pc
	endm

macro	op$7c			;ld a,h
	swap	d3
	move.b	d3,d7
	swap	d3
	next_pc
	endm

macro	op$7d			;ld a,l
	move.b	d3,d7
	next_pc
	endm

macro	op$7e			;ld a,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d7
	next_pc
	endm

macro	op$7f			;ld a,a
	next_pc
	endm

macro	op$80			;add a,b
	swap	d5
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	d5,d7
	move	sr,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$81			;add a,c
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	d5,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$82			;add a,d
	swap	d4
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	d4,d7
	move	sr,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$83			;add a,e
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	d4,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$84			;add a,h
	swap	d3
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	d3,d7
	move	sr,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$85			;add a,l
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	d3,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$86			;add a,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	(a3),d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$87			;add a,a
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	d7,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$88			;adc a,b
	swap	d5
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d5,d7
	move	sr,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$89			;adc a,c
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d5,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$8a			;adc a,d
	swap	d4
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d4,d7
	move	sr,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$8b			;adc a,e
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d4,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$8c			;adc a,h
	swap	d3
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d3,d7
	move	sr,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$8d			;adc a,l
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d3,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$8e			;adc a,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	d7,CPU_STATE_A_OLD(a4)
	move.b	(a3),d0
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d0,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$8f			;adc a,a
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d7,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$90			;sub b
	swap	d5
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	d5,d7
	move	sr,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$91			;sub c
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	d5,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$92			;sub d
	swap	d4
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	d4,d7
	move	sr,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$93			;sub e
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	d4,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$94			;sub h
	swap	d3
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	d3,d7
	move	sr,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$95			;sub l
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	d3,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$96			;sub (hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	(a3),d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$97			;sub a
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	d7,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$98			;sbc a,b
	swap	d5
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d5,d7
	move	sr,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$99			;sbc a,c
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d5,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$9a			;sbc a,d
	swap	d4
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d4,d7
	move	sr,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$9b			;sbc a,e
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d4,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$9c			;sbc a,h
	swap	d3
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d3,d7
	move	sr,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$9d			;sbc a,l
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d3,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$9e			;sbc a,(hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	d7,CPU_STATE_A_OLD(a4)
	move.b	(a3),d0
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d0,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$9f			;sbc a,a
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d7,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$a0			;and b
	swap	d5
	swap	d6
	and.b	d5,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$a1			;and c
	swap	d6
	and.b	d5,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$a2			;and d
	swap	d4
	swap	d6
	and.b	d4,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$a3			;and e
	swap	d6
	and.b	d4,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$a4			;and h
	swap	d3
	swap	d6
	and.b	d3,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$a5			;and l
	swap	d6
	and.b	d3,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$a6			;and (hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	swap	d6
	and.b	(a3),d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$a7			;and a
	swap	d6
	and.b	d7,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$a8			;xor b
	swap	d5
	swap	d6
	eor.b	d5,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$a9			;xor c
	swap	d6
	eor.b	d5,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$aa			;xor d
	swap	d4
	swap	d6
	eor.b	d4,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$ab			;xor e
	swap	d6
	eor.b	d4,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$ac			;xor h
	swap	d3
	swap	d6
	eor.b	d3,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$ad			;xor l
	swap	d6
	eor.b	d3,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$ae			;xor (hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	(a3),d0
	swap	d6
	eor.b	d0,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$af			;xor a
	swap	d6
	eor.b	d7,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$b0			;or b
	swap	d5
	swap	d6
	or.b	d5,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$b1			;or c
	swap	d6
	or.b	d5,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$b2			;or d
	swap	d4
	swap	d6
	or.b	d4,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$b3			;or e
	swap	d6
	or.b	d4,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$b4			;or h
	swap	d3
	swap	d6
	or.b	d3,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$b5			;or l
	swap	d6
	or.b	d3,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$b6			;or (hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	swap	d6
	or.b	(a3),d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$b7			;or a
	swap	d6
	or.b	d7,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$b8			;cp b
	move.b	d7,d0
	swap	d5
	swap	d6
	sub.b	d5,d0
	move	sr,d6
	swap	d6
	swap	d5
	next_pc
	endm

macro	op$b9			;cp c
	move.b	d7,d0
	swap	d6
	sub.b	d5,d0
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$ba			;cp d
	move.b	d7,d0
	swap	d4
	swap	d6
	sub.b	d4,d0
	move	sr,d6
	swap	d6
	swap	d4
	next_pc
	endm

macro	op$bb			;cp e
	move.b	d7,d0
	swap	d6
	sub.b	d4,d0
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$bc			;cp h
	move.b	d7,d0
	swap	d3
	swap	d6
	sub.b	d3,d0
	move	sr,d6
	swap	d6
	swap	d3
	next_pc
	endm

macro	op$bd			;cp l
	move.b	d7,d0
	swap	d6
	sub.b	d3,d0
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$be			;cp (hl)
	move.b	d3,d1
	move.l	d3,d2
	swap	d2
	mem_rd	d2
	move.b	d7,d0
	swap	d6
	sub.b	(a3),d0
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$bf			;cp a
	move.b	d7,d0
	swap	d6
	sub.b	d0,d0
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$c0			;ret nz
	local	quit
	btst	#FLAG_Z,d6
	bne.s	quit
	do_ret
quit:	next_pc
	endm

macro	op$c1			;pop bc
	move.b	(a1)+,d0
	move.b	(a1)+,d5
	swap	d5
	move.b	d0,d5
	sp_pop
	next_pc
	endm

macro	op$c2			;jp nz,##
	local	quit
	btst	#FLAG_Z,d6
	bne.s	quit
	do_jp
	move.l	a3,a0
	next_pc
quit:	addq.w	#2,a0
	next_pc
	endm

macro	op$c3			;jp ##
	do_jp
	move.l	a3,a0
	next_pc
	endm

macro	op$c4			;call nz,##
	local	quit
	lea	2(a0),a3
	btst	#FLAG_Z,d6
	bne.s	quit
	do_call
quit:	move.l	a3,a0
	next_pc
	endm

macro	op$c5			;push bc
	sp_push
	move.l	d5,d0
	swap	d0
	move.b	d0,-(a1)
	move.b	d5,-(a1)
	next_pc
	endm

macro	op$c6			;add a,#
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	add.b	(a0)+,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$c7			;rst 00
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	next_pc
	endm

macro	op$c8			;ret z
	local	quit
	btst	#FLAG_Z,d6
	beq.s	quit
	do_ret
quit:	next_pc
	endm

macro	op$c9			;ret
	do_ret
	next_pc
	endm

macro	op$ca			;jp z,##
	local	quit
	btst	#FLAG_Z,d6
	beq.s	quit
	do_jp
	move.l	a3,a0
	next_pc
quit:	addq.w	#2,a0
	next_pc
	endm

macro	op$cb			;prefix byte $cb
	local	jump
	move.b	(a0)+,jump+2-gbemu(a4)
jump:	jmp	OFFSET_CB+DONT_OPT_HI(a4)
	endm

macro	op$cc			;call z,##
	local	quit
	lea	2(a0),a3
	btst	#FLAG_Z,d6
	beq.s	quit
	do_call
quit:	move.l	a3,a0
	next_pc
	endm

macro	op$cd			;call ##
	lea	2(a0),a3
	do_call
	move.l	a3,a0
	next_pc
	endm

macro	op$ce			;adc a,#
	move.b	d7,CPU_STATE_A_OLD(a4)
	move.b	(a0)+,d0
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	addx.b	d0,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$cf			;rst 08
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	lea	$08(a0),a0
	next_pc
	endm

macro	op$d0			;ret nc
	local	quit
	btst	#FLAG_C,d6
	bne.s	quit
	do_ret
quit:	next_pc
	endm

macro	op$d1			;pop de
	move.b	(a1)+,d0
	move.b	(a1)+,d4
	swap	d4
	move.b	d0,d4
	sp_pop
	next_pc
	endm

macro	op$d2			;jp nc,##
	local	quit
	btst	#FLAG_C,d6
	bne.s	quit
	do_jp
	move.l	a3,a0
	next_pc
quit:	addq.w	#2,a0
	next_pc
	endm

macro	op$d3			;undefined
	return	UNKNOWN
	endm

macro	op$d4			;call nc,##
	local	quit
	lea	2(a0),a3
	btst	#FLAG_C,d6
	bne.s	quit
	do_call
quit:	move.l	a3,a0
	next_pc
	endm

macro	op$d5			;push de
	sp_push
	move.l	d4,d0
	swap	d0
	move.b	d0,-(a1)
	move.b	d4,-(a1)
	next_pc
	endm

macro	op$d6			;sub #
	move.b	d7,CPU_STATE_A_OLD(a4)
	swap	d6
	sub.b	(a0)+,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$d7			;rst 10
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	lea	$10(a0),a0
	next_pc
	endm

macro	op$d8			;ret c
	local	quit
	btst	#FLAG_C,d6
	beq.s	quit
	do_ret
quit:	next_pc
	endm

macro	op$d9			;reti
	moveq	#-1,d0
	move.b	d0,INT_IME(a4)
	and.b	INT_IE(a4),d0
	move.b	d0,INT_IM(a4)
	do_ret
	next_pc
	endm

macro	op$da			;jp c,##
	local	quit
	btst	#FLAG_C,d6
	beq.s	quit
	do_jp
	move.l	a3,a0
	next_pc
quit:	addq.w	#2,a0
	next_pc
	endm

macro	op$db			;undefined
	return	UNKNOWN
	endm

macro	op$dc			;call c,##
	local	quit
	lea	2(a0),a3
	btst	#FLAG_C,d6
	beq.s	quit
	do_call
quit:	move.l	a3,a0
	next_pc
	endm

macro	op$dd			;undefined
	return	UNKNOWN
	endm

macro	op$de			;sbc a,#
	move.b	d7,CPU_STATE_A_OLD(a4)
	move.b	(a0)+,d0
	swap	d6
	move	d6,ccr
	cmp.b	d7,d7
	subx.b	d0,d7
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$df			;rst 18
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	lea	$18(a0),a0
	next_pc
	endm

macro	op$e0			;ld (ff00+#),a
	move.b	d7,d0
	io_wr	(a0)+
	next_pc
	endm

macro	op$e1			;pop hl
	move.b	(a1)+,d0
	move.b	(a1)+,d3
	swap	d3
	move.b	d0,d3
	sp_pop
	next_pc
	endm

macro	op$e2			;ld (ff00+c),a
	move.b	d7,d0
	io_wr	d5
	next_pc
	endm

macro	op$e3			;undefined
	return	UNKNOWN
	endm

macro	op$e4			;undefined
	return	UNKNOWN
	endm

macro	op$e5			;push hl
	sp_push
	move.l	d3,d0
	swap	d0
	move.b	d0,-(a1)
	move.b	d3,-(a1)
	next_pc
	endm

macro	op$e6			;and #
	swap	d6
	and.b	(a0)+,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$e7			;rst 20
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	lea	$20(a0),a0
	next_pc
	endm

macro	op$e8			;add sp,#
	local	up,corr,go_on
	move.b	(a0)+,d0
	ext.w	d0
	bpl.s	up
	add.w	d0,a1
	add.b	d0,CPU_STATE_SP+1(a4)
	bcs.s	go_on
	subq.b	#1,CPU_STATE_SP+0(a4)
	bra.s	corr
up:	add.w	d0,a1
	add.b	d0,CPU_STATE_SP+1(a4)
	bcc.s	go_on
	addq.b	#1,CPU_STATE_SP+0(a4)
corr:	move.b	CPU_STATE_SP+1(a4),d1
	mem_rd	CPU_STATE_SP+0(a4)
	move.l	a2,CPU_STATE_SP_B(a4)
	move.l	a3,a1
go_on:	next_pc
	endm

macro	op$e9			;jp hl
	local	go_on
	move.l	d3,d2
	swap	d2
	move.b	d3,d1
	if	DEBUG & DEBUG_JP32K
	bmi.s	go_on
	addq.b	#1,d2
	subq.b	#1,d2
	bcc.s	go_on
	return	ERR_DEBUG
	endif
go_on:	mem_rd	d2
	move.l	a2,CPU_STATE_PC_B(a4)
	move.l	a3,a0
	next_pc
	endm

macro	op$ea			;ld (##),a
	move.b	d7,d0
	move.b	(a0)+,d1
	mem_wr	(a0)+
	next_pc
	endm

macro	op$eb			;undefined
	return	UNKNOWN
	endm

macro	op$ec			;undefined
	return	UNKNOWN
	endm

macro	op$ed			;undefined
	return	UNKNOWN
	endm

macro	op$ee			;xor #
	move.b	(a0)+,d0
	swap	d6
	eor.b	d0,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$ef			;rst 28
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	lea	$28(a0),a0
	next_pc
	endm

macro	op$f0			;ld a,(ff00+#)
	io_rd	(a0)+
	move.b	(a3),d7
	next_pc
	endm

macro	op$f1			;pop af
	local	nz,nc
	moveq	#FLAG_Z,d1
	moveq	#FLAG_C,d2
	bset	d1,d6
	bset	d2,d6
	move.b	(a1)+,d7
	moveq	#CPU_Z,d0
	and.b	d7,d0
	bne.s	nz
	bchg	d1,d6
nz:	moveq	#CPU_C,d0
	and.b	d7,d0
	bne.s	nc
	bchg	d2,d6
nc:	move.b	(a1)+,d7
	sp_pop
	next_pc
	endm

macro	op$f2			;ld a,(ff00+c)
	io_rd	d5
	move.b	(a3),d7
	next_pc
	endm

macro	op$f3			;di
	moveq	#0,d0
	move.b	d0,INT_IME(a4)
	move.b	d0,INT_IM(a4)
	next_pc
	endm

macro	op$f4			;undefined
	return	UNKNOWN
	endm

macro	op$f5			;push af
	local	nz,nc
	sp_push
	move.b	d7,-(a1)
	clr.b	d0
	btst	#FLAG_Z,d6
	beq.s	nz
	or	#CPU_Z,d0
nz:	btst	#FLAG_C,d6
	beq.s	nc
	or	#CPU_C,d0
nc:	move.b	d0,-(a1)
	next_pc
	endm

macro	op$f6			;or #
	swap	d6
	or.b	(a0)+,d7
	move	sr,d6
	and.b	#MASK_NC,d6
	swap	d6
	next_pc
	endm

macro	op$f7			;rst 30
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	lea	$30(a0),a0
	next_pc
	endm

macro	op$f8			;ld hl,sp+#
	move.b	(a0)+,d0
	ext.w	d0
	add.w	CPU_STATE_SP(a4),d0
	move.w	d0,-(sp)
	move.b	(sp)+,d3
	swap	d3
	move.b	d0,d3
	next_pc
	endm

macro	op$f9			;ld sp,hl
	move.b	d3,d1
	move.b	d1,CPU_STATE_SP+1(a4)
	move.l	d3,d2
	swap	d2
	move.b	d2,CPU_STATE_SP+0(a4)
	mem_rd	d2
	move.l	a2,CPU_STATE_SP_B(a4)
	move.l	a3,a1
	next_pc
	endm

macro	op$fa			;ld a,(##)
	move.b	(a0)+,d1
	mem_rd	(a0)+
	move.b	(a3),d7
	next_pc
	endm

macro	op$fb			;ei
	local	go_on,jump
	dbra	d6,go_on
	jsr	V_LCD_LINE(a4)
go_on:	moveq	#-1,d0
	move.b	d0,INT_IME(a4)
	and.b	INT_IE(a4),d0
	move.b	d0,INT_IM(a4)
	if	DEBUG & DEBUG_STEP
	return	SINGLE_STEP
	else
	move.b	(a0)+,jump+2-gbemu(a4)
jump:	jmp	OFFSET_OP+DONT_OPT_HI(a4)
	endif
	endm

macro	op$fc			;undefined
	return	UNKNOWN
	endm

macro	op$fd			;undefined
	return	UNKNOWN
	endm

macro	op$fe			;cp #
	move.b	d7,d0
	swap	d6
	sub.b	(a0)+,d0
	move	sr,d6
	swap	d6
	next_pc
	endm

macro	op$ff			;rst 38
	do_push	a0
	move.l	BASE_ROM_0(a4),a0
	move.l	a0,CPU_STATE_PC_B(a4)
	lea	$38(a0),a0
	next_pc
	endm

;================================================================
;	end of include
