/****************************************************************
 *  emulate.c
 *			High level functions of emulation
 *
 *  2000-02-29  Bodo Wenzel  Creation in simple version
 *  2000-03-07  Bodo Wenzel  Creation of user interface in test
 *  2000-04-28  Bodo Wenzel  Copied to full version
 *  2000-05-06  Bodo Wenzel  External RAM pageable
 *  2000-06-18  Bodo Wenzel  New memory management
 *  2000-08-29  Bodo Wenzel  Much more support for breakpoints
 *  2000-09-09  Bodo Wenzel  Now nearly full screen emulation,
 *                           added timing for profile window
 *  2000-09-30  Bodo Wenzel  PalmBoy scroller as GB code
 *  2000-10-15  Bodo Wenzel  Interrupts and timer,
 *                           better breakpoints
 *  2000-11-18  Bodo Wenzel  New screen rendering and
 *                           better joypad emulation
 *  2001-01-21  Bodo Wenzel  Joystick emulation via keys
 *  2001-02-04  Bodo Wenzel  New stack management
 *  2001-02-10  Bodo Wenzel  Debug enhancements
 *  2001-02-18  Bodo Wenzel  Support of more cartridges
 *  2001-02-18  Bodo Wenzel  Last time debugged (touched DEBUG)
 *  2001-06-20  M. Jaap      Alternate key for "START": Return
 ****************************************************************

  (c)2000 Bodo Wenzel

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 ****************************************************************/

/* === Includes ===============================================	*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aes.h>
#include <tos.h>

#include "emulate.h"
#include "stemboy.h"
#include "cartridg.h"
#include "gem_add.h"
#include "emu_wrap.h"
#include "support.h"

#define  noDEBUG  /* define for _NOT_ hooking ikbdsys() */

/* === Constants ==============================================	*/

#define  OPCODE_BP  (0xdb)  /* opcode for "de break" */

/* === Variables ==============================================	*/

EMU_STATE  emu_state;

clock_t  emu_time;
long     emu_frames;

unsigned char  *emu_rom=NULL;
unsigned char  *emu_rom_pages[MAX_ROM_BANKS];
unsigned char  emu_ram_ext[0x8000L];
unsigned char  *emu_ram_pages[MAX_RAM_BANKS];
void  (*emu_v_ram_enable)(void);
void  (*emu_v_roml_select)(void);
void  (*emu_v_romh_select)(void);
void  (*emu_v_ram_select)(void);
void  (*emu_v_mode_select)(void);

unsigned char  emu_direction;
unsigned char  emu_command;
char           emu_quit;

static  KBDVBASE                 *vbase;
static  void                     (*old_kbdsys)(void);
static  enum {FIRST=0,MX,MY,JS}  kbd_state;

static  enum { RUN=0, RESET, TRY_SCROLLER, TRY_CODE }
  start_mode;

static  unsigned char  ram_vid[0x2000L];
static  unsigned char  ram_int[0x2000L];
static  unsigned char  ram_oam[0x0100L];
static  unsigned char  ram_cpu[0x0080L];

static  unsigned char  *bp_addr[10];
static  unsigned char  bp_org_opcode[10];
static  int            bp_last;

/* === Prototypes =============================================	*/

static void my_kbdsys(void);

/* === Functions ==============================================	*/

void emu_init(void) {
  /*---------------*/
  OBSPEC    *alert;
  int       i;

  emu_direction=0xef;
  emu_command=0xdf;

  /* just to enable "Run" and some safety after reset */
  emu_rom=(unsigned char *)malloc(0x8000);
  if (emu_rom==NULL) {
    rsrc_gaddr(R_FRSTR,AMEMERR,&alert);
    form_alert(1,alert->free_string);
  } else {
    memset(emu_rom,0x00,0x8000);
    emu_rom[0x0100]=0x18;
    emu_rom[0x0101]=0xfe;
    emu_rom[0x7ffd]=0xc3;
    emu_rom[0x7ffe]=0x00;
    emu_rom[0x7fff]=0x01;
  }
  for (i=0; i<MAX_ROM_BANKS; i++)
    emu_rom_pages[i]=emu_rom;
  emu_v_ram_enable =mbc_dummy;
  emu_v_roml_select=mbc_dummy;
  emu_v_romh_select=mbc_dummy;
  emu_v_ram_select =mbc_dummy;
  emu_v_mode_select=mbc_dummy;

  for (i=0; i<10; i++)
    bp_addr[i]=NULL;
  bp_last=-1;
}

void emu_exit(void) {
  /*---------------*/
  if (emu_rom!=NULL)
    free(emu_rom);
}

void emu_mykbd(void) {
  /*----------------*/
  while (Kbshift(-1)&0x2f);
  kbd_state=FIRST;
  emu_quit=0;

  vbase=Kbdvbase();
  old_kbdsys=vbase->kb_kbdsys;
  vbase->kb_kbdsys=my_kbdsys;
}

void emu_oldkbd(void) {
  /*-----------------*/
  vbase->kb_kbdsys=old_kbdsys;
}

void emu_reset(EMU_COMBO *emulation) {
  /*--------------------------------*/
  emulation[V_OPCODE_STOP].ext.vector.addr=opcode_stop;

  emulation[LCD].ext.lcd_regs.lcdc=0x80;
  emulation[LCD].ext.lcd_regs.i_ly=-1;
  emulation[LCD].ext.lcd_regs.i_wy=0;
  emulation[LCD].ext.lcd_regs.bgp =0x00;
  emulation[LCD].ext.lcd_regs.obp0=0x00;
  emulation[LCD].ext.lcd_regs.obp1=0x00;

  emulation[V_IO_WR_LCDC].ext.vector.addr=io_wr_lcdc;
  emulation[V_IO_RD_STAT].ext.vector.addr=io_rd_stat;
  emulation[V_IO_WR_STAT].ext.vector.addr=io_wr_stat;
  emulation[V_IO_WR_SCY].ext.vector.addr =io_wr_scy;
  emulation[V_IO_WR_SCX].ext.vector.addr =io_wr_scx;
  emulation[V_IO_WR_LYC].ext.vector.addr =io_wr_lyc;
  emulation[V_IO_WR_DMA].ext.vector.addr =io_wr_dma;
  emulation[V_IO_WR_BGP].ext.vector.addr =io_wr_bgp;
  emulation[V_IO_WR_OBP0].ext.vector.addr=io_wr_obp0;
  emulation[V_IO_WR_OBP1].ext.vector.addr=io_wr_obp1;
  emulation[V_IO_WR_WY].ext.vector.addr  =io_wr_wy;
  emulation[V_IO_WR_WX].ext.vector.addr  =io_wr_wx;

  emulation[BASE_RAM].ext.banks_ram.ram_vid =ram_vid;
  emulation[BASE_RAM].ext.banks_ram.ram_vid+=0x10000L-0x8000L;
  emulation[BASE_RAM].ext.banks_ram.ram_int =ram_int;
  emulation[BASE_RAM].ext.banks_ram.ram_int+=0x10000L-0xc000L;
  emulation[BASE_RAM].ext.banks_ram.ram_oam =ram_oam;
  emulation[BASE_RAM].ext.banks_ram.ram_oam+=0x10000L-0xfe00L;
  emulation[BASE_RAM].ext.banks_ram.ram_cpu =ram_cpu;
  emulation[BASE_RAM].ext.banks_ram.ram_cpu+=0x10000L-0xff80L;
  memset(ram_vid,0x00,0x1800);
  memset(ram_vid+0x1800,0xff,0x0800);
  memset(ram_int,0x00,0x2000);
  memset(ram_oam,0x00,0x0100);
  memset(ram_cpu,0x00,0x0080);

  emulation[BASE_ROM].ext.banks_rom.rom_0=
    emulation[CODE_RESET].ext.gb_code;
  emulation[CPU_STATE].ext.cpu_regs.pc_base=
    emulation[CODE_RESET].ext.gb_code;
  emulation[CPU_STATE].ext.cpu_regs.pc=0;
  emulation[TIMING].ext.timing_vars.cycles=32000; /* more than enough */
  while (gbrun(emulation)==SINGLE_STEP);

  start_mode=RESET;
}

void emu_run(EMU_COMBO *emulation) {
  /*------------------------------*/
  void            (*v_lcd_line)(void);
  unsigned short  cycles;
  int             i;
  unsigned char   *pc;

  if (start_mode!=RUN) {
    if (start_mode==RESET)
      emulation[CPU_STATE].ext.cpu_regs.pc=0x0100;
    else if (start_mode==TRY_SCROLLER)
      emulation[CPU_STATE].ext.cpu_regs.pc=CODE_SCROLLER*0x100;
    else {
      emulation[CPU_STATE].ext.cpu_regs.a =0xef;
      emulation[CPU_STATE].ext.cpu_regs.f =0x00;
      emulation[CPU_STATE].ext.cpu_regs.b =0x01;
      emulation[CPU_STATE].ext.cpu_regs.c =0x23;
      emulation[CPU_STATE].ext.cpu_regs.d =0x45;
      emulation[CPU_STATE].ext.cpu_regs.e =0x67;
      emulation[CPU_STATE].ext.cpu_regs.h =0x89;
      emulation[CPU_STATE].ext.cpu_regs.l =0xab;
      emulation[CPU_STATE].ext.cpu_regs.pc=0x0000;
    }
    emulation[CPU_STATE].ext.cpu_regs.pc_base=emu_rom;
    emulation[BASE_ROM].ext.banks_rom.rom_0=emu_rom;
    emulation[BASE_ROM].ext.banks_rom.rom_1=emu_rom_pages[0];
    emulation[BASE_RAM].ext.banks_ram.ram_ext=emu_ram_pages[0];
    emulation[V_RAM_ENABLE].ext.vector.addr =emu_v_ram_enable;
    emulation[V_ROML_SELECT].ext.vector.addr=emu_v_roml_select;
    emulation[V_ROMH_SELECT].ext.vector.addr=emu_v_romh_select;
    emulation[V_RAM_SELECT].ext.vector.addr =emu_v_ram_select;
    emulation[V_MODE_SELECT].ext.vector.addr=emu_v_mode_select;
    emulation[V_RAM_RD_EXT].ext.vector.addr=
      (void (*)())&(emulation[RAM_RD_EXT_STD].ext.func);
    emulation[V_RAM_WR_EXT].ext.vector.addr=
      (void (*)())&(emulation[RAM_WR_EXT_STD].ext.func);

    emulation[TIMING].ext.timing_vars.cycles=0;
    emulation[INT_VARIABLES].ext.int_vars.ime=0;
    emulation[INT_VARIABLES].ext.int_vars.ie=0;
    emulation[INT_VARIABLES].ext.int_vars.ifl=0;
    emulation[INT_VARIABLES].ext.int_vars.im=0;

    emulation[JOYPAD].ext.joypad.both=0xcf;
    emulation[JOYPAD].ext.joypad.cmd=0xdf;
    emulation[JOYPAD].ext.joypad.dir=0xef;
    emulation[JOYPAD].ext.joypad.none=0xff;

    start_mode=RUN;
    bp_last=-1;
  }
  emu_state.run_result=0;

  if (bp_last>=0) {
    v_lcd_line=emulation[V_LCD_LINE].ext.vector.addr;
    emulation[V_LCD_LINE].ext.vector.addr=
      (void (*)())&(emulation[BP_RTS].ext.func);
    cycles=emulation[TIMING].ext.timing_vars.cycles;
    emulation[TIMING].ext.timing_vars.cycles=0;

    emu_state.run_result=gbrun(emulation);

    if (emulation[V_LCD_LINE].ext.vector.addr==
        (void (*)())&(emulation[BP_RTS].ext.func)) {
      emulation[TIMING].ext.timing_vars.cycles=cycles;
      emulation[V_LCD_LINE].ext.vector.addr=v_lcd_line;
    }
    bp_last=-1;
  }

  if (emu_state.run_result==0) {
    for (i=0; i<10; i++)
      if (bp_addr[i]!=NULL) {
        bp_org_opcode[i]=*bp_addr[i];
        *bp_addr[i]=OPCODE_BP;
      }

    emu_time-=clock();
    emu_state.run_result=gbrun(emulation);
    emu_time+=clock();

    for (i=9; i>=0; i--)
      if (bp_addr[i]!=NULL)
        *bp_addr[i]=bp_org_opcode[i];
    if (emu_state.run_result==UNKNOWN) {
      pc=emulation[CPU_STATE].ext.cpu_regs.pc_base+
         emulation[CPU_STATE].ext.cpu_regs.pc-1;
      for (i=0; i<10; i++)
        if (bp_addr[i]==pc) {
          emulation[CPU_STATE].ext.cpu_regs.pc--;
          if (emulation[OPCODE_BP-0x100].prof!=0)
            emulation[OPCODE_BP-0x100].prof--;
          emu_state.run_result=i;
          bp_last=i;
          break;
        }
    }
  }

  emu_state.int_enabled=emulation[INT_VARIABLES].ext.int_vars.ime;
}

void emu_try_scroller(EMU_COMBO *emulation) {
  /*---------------------------------------*/
  OBSPEC  *alert;
  int     msg[8];

  emu_rom=(unsigned char *)realloc(emu_rom,0x8000L);
  if (emu_rom==NULL) {
    rsrc_gaddr(R_FRSTR,AMEMERR,&alert);
    form_alert(1,alert->free_string);
    return;
  }

  cartr_no_info();

  memcpy(emu_rom,emulation[CODE_RESET].ext.gb_code,0x8000L);

  emu_rom_pages[0]=emu_rom;
  emu_rom_pages[1]=emu_rom;

  emu_ram_pages[0]=emu_ram_ext+(0x10000L-0xa000L);

  emu_v_ram_enable =mbc_dummy;
  emu_v_roml_select=mbc_dummy;
  emu_v_romh_select=mbc_dummy;
  emu_v_ram_select =mbc_dummy;
  emu_v_mode_select=mbc_dummy;

  start_mode=TRY_SCROLLER;

  msg[0]=MN_SELECTED;
  msg[1]=ap_id;
  msg[2]=0;
  msg[3]=MEMU;
  msg[4]=MERUN;
  menu_tnormal(menu,msg[3],0);
  appl_write(ap_id,16,msg);
}

void emu_try_code(EMU_COMBO *emulation,int kchar) {
  /*---------------------------------------------*/
  OBJECT  *o;
  int     x,y,w,h,b;
  char    *bak,*s;
  OBSPEC  *alert;
  int     i,a;
  int     msg[8];

  (void)emulation;  /* may be used possibly */

  rsrc_gaddr(R_TREE,DTRY,&o);
  bak=strdup(o[DTOPCODE].ob_spec.tedinfo->te_ptext);
  if (bak==NULL)
    return;

  form_center(o,&x,&y,&w,&h);
  form_dial(FMD_START,0,0,0,0,x,y,w,h);
  objc_draw(o,0,99,x,y,w,h);

  if (kchar!='\0') {
    objc_edit(o,DTOPCODE,0,&x,ED_INIT);
    objc_edit(o,DTOPCODE,kchar,&x,ED_CHAR);
    objc_edit(o,DTOPCODE,0,&x,ED_END);
  }

  b=form_do(o,DTOPCODE);
  objc_change(o,b,0,x,y,w,h,NORMAL,1);
  form_dial(FMD_FINISH,0,0,0,0,x,y,w,h);

  if (b==DTABORT) {
    strcpy(o[DTOPCODE].ob_spec.tedinfo->te_ptext,bak);
    free(bak);
    return;
  }
  free(bak);

  emu_rom=(unsigned char *)realloc(emu_rom,0x20000L);
  if (emu_rom==NULL) {
    rsrc_gaddr(R_FRSTR,AMEMERR,&alert);
    form_alert(1,alert->free_string);
    return;
  }

  cartr_no_info();

  memset(emu_rom+0x00000L,0x10,0x4000);
  memset(emu_rom+0x04000L,0xcf,0x4000);
  memset(emu_rom+0x08000L,0xd7,0x4000);
  memset(emu_rom+0x0c000L,0xdf,0x4000);
  memset(emu_rom+0x10000L,0xe7,0x4000);
  memset(emu_rom+0x14000L,0xef,0x4000);
  memset(emu_rom+0x18000L,0xf7,0x4000);
  memset(emu_rom+0x1c000L,0xff,0x4000);

  s=o[DTOPCODE].ob_spec.tedinfo->te_ptext;
  a=0x0000;
  while (s[0]!='\0' && s[1]!='\0') {
    if (s[0]>='0' && s[0]<='9')
      emu_rom[a]=s[0]-'0';
    else if (s[0]>='A' && s[0]<='F')
      emu_rom[a]=s[0]-'A'+0xa;
    else if (s[0]>='a' && s[0]<='f')
      emu_rom[a]=s[0]-'a'+0xa;
    emu_rom[a]<<=4;
    if (s[1]>='0' && s[1]<='9')
      emu_rom[a]|=s[1]-'0';
    else if (s[1]>='A' && s[1]<='F')
      emu_rom[a]|=s[1]-'A'+0xa;
    else if (s[1]>='a' && s[1]<='f')
      emu_rom[a]|=s[1]-'a'+0xa;
    a++;
    s+=2;
  }
  emu_rom[a++]=0x10;
  emu_rom[a++]=0x10;
  emu_rom[a++]=0x10;
  memcpy(emu_rom+0x04000L,emu_rom,a);
  memcpy(emu_rom+0x08000L,emu_rom,a);
  memcpy(emu_rom+0x0c000L,emu_rom,a);
  memcpy(emu_rom+0x10000L,emu_rom,a);
  memcpy(emu_rom+0x14000L,emu_rom,a);
  memcpy(emu_rom+0x18000L,emu_rom,a);
  memcpy(emu_rom+0x1c000L,emu_rom,a);

  emu_rom_pages[0]=emu_rom;
  for (i=1; i<MAX_ROM_BANKS; i++)
    emu_rom_pages[i]=emu_rom+(i%8)*0x4000L-0x4000L;

  memset(ram_vid,0x11,0x2000);
  memset(ram_int,0x33,0x2000);
  memset(ram_oam,0x44,0x0100);
  memset(ram_cpu,0x55,0x0080);

  memset(emu_ram_ext+0x0000L,0x21,0x2000);
  memset(emu_ram_ext+0x2000L,0x22,0x2000);
  memset(emu_ram_ext+0x4000L,0x23,0x2000);
  memset(emu_ram_ext+0x6000L,0x24,0x2000);

  emu_ram_pages[0]=emu_ram_ext+(0x10000L-0xa000L)+0x0000L;
  emu_ram_pages[1]=emu_ram_ext+(0x10000L-0xa000L)+0x2000L;
  emu_ram_pages[2]=emu_ram_ext+(0x10000L-0xa000L)+0x4000L;
  emu_ram_pages[3]=emu_ram_ext+(0x10000L-0xa000L)+0x6000L;

  emu_v_ram_enable =mbc_dummy;
  emu_v_roml_select=mbc3_rom_select;
  emu_v_romh_select=mbc3_rom_select;
  emu_v_ram_select =mbc3_ram_select;
  emu_v_mode_select=mbc3_rtc_latch;

  start_mode=TRY_CODE;

  msg[0]=MN_SELECTED;
  msg[1]=ap_id;
  msg[2]=0;
  msg[3]=MEMU;
  msg[4]=MERUN;
  menu_tnormal(menu,msg[3],0);
  appl_write(ap_id,16,msg);
}

void emu_set_bp(void) {
  /*-----------------*/
  char    bak[10][10];
  OBJECT  *o;
  int     x,y,w,h,b,i,oi;
  long    a;

  if (emu_rom==NULL)
    return;

  rsrc_gaddr(R_TREE,DBREAKP,&o);
  strcpy(bak[0],o[DBPADDR0].ob_spec.tedinfo->te_ptext);
  strcpy(bak[1],o[DBPADDR1].ob_spec.tedinfo->te_ptext);
  strcpy(bak[2],o[DBPADDR2].ob_spec.tedinfo->te_ptext);
  strcpy(bak[3],o[DBPADDR3].ob_spec.tedinfo->te_ptext);
  strcpy(bak[4],o[DBPADDR4].ob_spec.tedinfo->te_ptext);
  strcpy(bak[5],o[DBPADDR5].ob_spec.tedinfo->te_ptext);
  strcpy(bak[6],o[DBPADDR6].ob_spec.tedinfo->te_ptext);
  strcpy(bak[7],o[DBPADDR7].ob_spec.tedinfo->te_ptext);
  strcpy(bak[8],o[DBPADDR8].ob_spec.tedinfo->te_ptext);
  strcpy(bak[9],o[DBPADDR9].ob_spec.tedinfo->te_ptext);

  form_center(o,&x,&y,&w,&h);
  form_dial(FMD_START,0,0,0,0,x,y,w,h);
  objc_draw(o,0,99,x,y,w,h);
  b=form_do(o,DBPADDR0);
  objc_change(o,b,0,x,y,w,h,NORMAL,1);
  form_dial(FMD_FINISH,0,0,0,0,x,y,w,h);

  if (b==DBPABORT) {
    strcpy(o[DBPADDR0].ob_spec.tedinfo->te_ptext,bak[0]);
    strcpy(o[DBPADDR1].ob_spec.tedinfo->te_ptext,bak[1]);
    strcpy(o[DBPADDR2].ob_spec.tedinfo->te_ptext,bak[2]);
    strcpy(o[DBPADDR3].ob_spec.tedinfo->te_ptext,bak[3]);
    strcpy(o[DBPADDR4].ob_spec.tedinfo->te_ptext,bak[4]);
    strcpy(o[DBPADDR5].ob_spec.tedinfo->te_ptext,bak[5]);
    strcpy(o[DBPADDR6].ob_spec.tedinfo->te_ptext,bak[6]);
    strcpy(o[DBPADDR7].ob_spec.tedinfo->te_ptext,bak[7]);
    strcpy(o[DBPADDR8].ob_spec.tedinfo->te_ptext,bak[8]);
    strcpy(o[DBPADDR9].ob_spec.tedinfo->te_ptext,bak[9]);
    return;
  }

  for (i=0; i<10; i++) {
    switch (i) {
    case 1:   oi=DBPADDR1;  break;
    case 2:   oi=DBPADDR2;  break;
    case 3:   oi=DBPADDR3;  break;
    case 4:   oi=DBPADDR4;  break;
    case 5:   oi=DBPADDR5;  break;
    case 6:   oi=DBPADDR6;  break;
    case 7:   oi=DBPADDR7;  break;
    case 8:   oi=DBPADDR8;  break;
    case 9:   oi=DBPADDR9;  break;
    default:  oi=DBPADDR0;  break; }
    if (sscanf(o[oi].ob_spec.tedinfo->te_ptext,"%lX",&a)!=1)
      bp_addr[i]=NULL;
    else if (a>=emu_rom_pages[MAX_ROM_BANKS-1]-emu_rom+0x8000L)
      bp_addr[i]=NULL;
    else
      bp_addr[i]=emu_rom+a;
  }
  bp_last=-1;
}

static long warmreset(void) {
  /*-----------------------*/
  (*( (void (*)(void)) (* (long *)0x4F2) ))();
  return(0);			/* does never return... */
}

static long coldreset(void) {
  /*-----------------------*/
  *((long *)0x420)=0;
  *((long *)0x43A)=0;
  (*( (void (*)(void)) (* (long *)0x4F2) ))();
  return(0);			/* does never return... */
}

static void my_kbdsys(void) {
  /*-----------------------*/
#ifdef DEBUG

  /* a really primitive way for debugging */
  (*old_kbdsys)();
  emu_quit=-1;

#else

  register unsigned char  st;
  register unsigned char  da;

  st=*((unsigned char *)0xFFFFFC00L);
  da=*((unsigned char *)0xFFFFFC02L);
  if (st&0x70)			/* nearly no error service... */
    return;

  if (kbd_state==MX) {		/* second mouse byte */
    /* ignore mx+=(tm_byte)da; */
    kbd_state=MY;
  } else if (kbd_state==MY) {	/* third mouse byte */
    /* ignore my+=(tm_byte)da; */
    kbd_state=FIRST;

  } else if (kbd_state==JS) {	/* second joystick byte */
    if (da&0x01)
      emu_direction&=~0x4;
    else
      emu_direction|=0x4;
    if (da&0x02)
      emu_direction&=~0x8;
    else
      emu_direction|=0x8;
    if (da&0x04)
      emu_direction&=~0x2;
    else
      emu_direction|=0x2;
    if (da&0x08)
      emu_direction&=~0x1;
    else
      emu_direction|=0x1;
    /* the fire key is reported by mouse packets */
    kbd_state=FIRST;

  } else {			/* first byte */
    kbd_state=FIRST;
    if (da&0x80) {		/* special / key released */
      switch (da) {
      case 0xFE:
      case 0xFF:
        kbd_state=JS;
        return;
      case 0xFB:
      case 0xFA:
        kbd_state=MX;
        return;
      case 0xF9:
        emu_command&=~0x1;
        kbd_state=MX;
        return;
      case 0xF8:
        emu_command|=0x1;
        kbd_state=MX;
        return;
      case 0xB8:		/* ALT */
        emu_command|=0x1;
        return;
      case 0xAA:		/* SHIFT-L */
        emu_command|=0x2;
        return;
      case 0xB6:		/* SHIFT-R */
        emu_command|=0x8;
        return;
      case 0x9C:		/* RETURN */
        emu_command|=0x8;
        return;
      case 0x9D:		/* CTRL */
        emu_command|=0x4;
        return;
      case 0xC8:		/* UP */
        emu_direction|=0x4;
        return;
      case 0xD0:		/* DOWN */
        emu_direction|=0x8;
        return;
      case 0xCB:		/* LEFT */
        emu_direction|=0x2;
        return;
      case 0xCD:		/* RIGHT */
        emu_direction|=0x1;
        return;
      case 0x81:		/* ESC */
        emu_quit=1;
      default:
        return;
      }

    } else {			/* key pressed */
      switch (da) {
      case 0x38:		/* ALT */
        emu_command&=~0x1;
        return;
      case 0x2A:		/* SHIFT-L */
        emu_command&=~0x2;
        return;
      case 0x36:		/* SHIFT-R */
        emu_command&=~0x8;
        return;
      case 0x1C:		/* SHIFT-R */
        emu_command&=~0x8;
        return;
      case 0x1D:		/* CTRL */
        emu_command&=~0x4;
        return;
      case 0x48:		/* UP */
        emu_direction&=~0x4;
        return;
      case 0x50:		/* DOWN */
        emu_direction&=~0x8;
        return;
      case 0x4B:		/* LEFT */
        emu_direction&=~0x2;
        return;
      case 0x4D:		/* RIGHT */
        emu_direction&=~0x1;
        return;
      case 0x53:		/* DELETE */
        if ((emu_command&(0x1|0x4))==0)
          return;
        if (emu_command&0x8)
          Supexec(warmreset);
        else
          Supexec(coldreset);	/* never returns... */
      default:
        return;
      }
    }
  }

#endif
}

/* === End ====================================================	*/
