/****************************************************************
 *  cartridg.c
 *			Functions for GB cartridge files
 *
 *  2000-02-29  Bodo Wenzel  Creation in simple version
 *  2000-03-07  Bodo Wenzel  Creation in test
 *  2000-04-28  Bodo Wenzel  Copied to full version
 *  2000-08-22  Bodo Wenzel  Quickstart via commandline
 *  2000-09-29  Bodo Wenzel  Free emu_rom if load error
 *  2001-02-18  Bodo Wenzel  Support of more cartridges
 ****************************************************************

  (c)2000 Bodo Wenzel

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 ****************************************************************/

/* === Includes ===============================================	*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aes.h>
#include <ext.h>

#include "cartridg.h"
#include "stemboy.h"
#include "emulate.h"
#include "support.h"

/* === Constants ==============================================	*/

#define  FALSE  (0)
#define  TRUE   (!FALSE)

#define  MAX_ROM  (MAX_ROM_BANKS*0x4000L)
#define  MAX_RAM  (MAX_RAM_BANKS*0x2000L)

#define  EXT  "GB?"

/* === Variables ==============================================	*/

typedef  struct {
  unsigned char  unused1[0x134];
  char           name[16];
  unsigned char  unused2[3];
  unsigned char  ctype;
  unsigned char  romsz;
  unsigned char  ramsz;
}  CARTR_HEADER;

typedef  struct {
  char  name[17];
  char  *cname;
  long  romsz;
  long  ramsz;
  long  fsz;
}  HEADER;

static  HEADER  header={ "","",0,0,0 };

typedef  struct {
  int   id;
  int   mbc;
  char  *name;
} CARTR_TYPE;

static  CARTR_TYPE  known_types[]={
  { 0x00, 0,"ROM ONLY" },
  { 0x01, 1,"MBC1" },
  { 0x02, 1,"MBC1+RAM" },
  { 0x03, 1,"MBC1+RAM+BAT" },
  { 0x05, 2,"MBC2" },
  { 0x06, 2,"MBC2+BAT" },
  { 0x08, 0,"RAM" },
  { 0x09, 0,"RAM+BAT" },
  { 0x0B,-1,"MMM01" },
  { 0x0C,-1,"MMM01+RAM" },
  { 0x0D,-1,"MMM01+RAM+BAT" },
  { 0x0F, 3,"MBC3+BAT+RTC" },
  { 0x10, 3,"MBC3+RAM+BAT+RTC" },
  { 0x11, 3,"MBC3" },
  { 0x12, 3,"MBC3+RAM" },
  { 0x13, 3,"MBC3+RAM+BAT" },
  { 0x15,-1,"MBC4" },
  { 0x16,-1,"MBC4+RAM" },
  { 0x17,-1,"MBC4+RAM+BAT" },
  { 0x19, 5,"MBC5" },
  { 0x1A, 5,"MBC5+RAM" },
  { 0x1B, 5,"MBC5+RAM+BAT" },
  { 0x1C, 5,"MBC5+MOT" },
  { 0x1D, 5,"MBC5+RAM+MOT" },
  { 0x1E, 5,"MBC5+RAM+BAT+MOT" },
  { 0xFC,-1,"Nintendo Pocket Camera" },
  { 0xFD,-1,"Bandai TAMA5" },
  { 0xFE,-1,"Hudson HuC-3" },
  { 0xFF,-1,"Hudson HuC-1" },
  { -1,  -1,"?" }
};

/* === Functions ==============================================	*/

int cartr_load(char *filename) {
  /*--------------------------*/
  static char          fp[PATH_MAX+10]="";
  static char          fn[20]="";
         char          n[PATH_MAX+30];
         char          *p;
         int           b;
         FILE          *f;
         OBSPEC        *alert;
         CARTR_HEADER  h;
         int           i,mbc,rom_banks;

  if (filename!=NULL)
    strcpy(n,filename);
  else {
    if (fp[0]=='\0') {
      if (getcwd(fp,PATH_MAX)==NULL)
        fp[0]='\0';
      b=(int)(strlen(fp)-1);
      if (fp[b]=='\\')
        fp[b]='\0';
      strcat(fp,"\\*." EXT);
    }

    if (!fsel_input(fp,fn,&b))
      b=0;
    if (b==0 || fn[0]=='\0')
      return FALSE;

    strcpy(n,fp);
    if ((p=strrchr(n,'\\'))==NULL)
      p=n;
    else
      p++;
    strcpy(p,fn);
  }

  f=fopen(n,"rb");
  if (f==NULL) {
    rsrc_gaddr(R_FRSTR,AFRDERR,&alert);
    form_alert(1,alert->free_string);
    return FALSE;
  }

  if (fread(&h,sizeof(CARTR_HEADER),1,f)!=1) {
    fclose(f);
    rsrc_gaddr(R_FRSTR,AFRDERR,&alert);
    form_alert(1,alert->free_string);
    return FALSE;
  }

  strncpy(header.name,h.name,16);
  header.name[16]='\0';

  for (i=0; known_types[i].id>=0 &&
            known_types[i].id!=h.ctype; i++);
  header.cname=known_types[i].name;
  mbc=known_types[i].mbc;

  rom_banks=2<<(h.romsz&0x0f);
  if ((h.romsz&0xf0)!=0)
    rom_banks+=2<<(h.romsz>>4);
  header.romsz=rom_banks*0x4000L;

  switch (h.ctype) {
  case 0x05:
  case 0x06:
    header.ramsz=0x200L/2;
    break;
  case 0xfe:
  case 0xff:
    header.ramsz=0x8000L;
    break;
  default:
    header.ramsz=(h.ramsz==0) ? 0 : 0x200L<<(2*h.ramsz);
    break;
  }

  fseek(f,0,SEEK_END);
  header.fsz=ftell(f);
  fseek(f,0,SEEK_SET);

  if (mbc<0) {
    fclose(f);
    rsrc_gaddr(R_FRSTR,ACTUNSUP,&alert);
    form_alert(1,alert->free_string);
    return FALSE;
  }

  if (header.romsz!=header.fsz) {
    fclose(f);
    rsrc_gaddr(R_FRSTR,AFSWRONG,&alert);
    form_alert(1,alert->free_string);
    return FALSE;
  }

  emu_rom=(unsigned char *)realloc(emu_rom,header.fsz);
  if (emu_rom==NULL) {
    fclose(f);
    rsrc_gaddr(R_FRSTR,AMEMERR,&alert);
    form_alert(1,alert->free_string);
    return FALSE;
  }

  if (fread(emu_rom,sizeof(unsigned char),header.fsz,f)
      !=header.fsz) {
    fclose(f);
    free(emu_rom);
    emu_rom=NULL;
    rsrc_gaddr(R_FRSTR,AFRDERR,&alert);
    form_alert(1,alert->free_string);
    return FALSE;
  }
  fclose(f);

  emu_rom_pages[0]=emu_rom;
  for (i=1; i<MAX_ROM_BANKS; i++)
    emu_rom_pages[i]=emu_rom+(i%rom_banks)*0x4000L-0x4000L;

  if (header.ramsz>0x2000L) {
    emu_ram_pages[0]=emu_ram_ext+(0x10000L-0xa000L)+0x0000L;
    emu_ram_pages[1]=emu_ram_ext+(0x10000L-0xa000L)+0x2000L;
    emu_ram_pages[2]=emu_ram_ext+(0x10000L-0xa000L)+0x4000L;
    emu_ram_pages[3]=emu_ram_ext+(0x10000L-0xa000L)+0x6000L;
  } else {
    emu_ram_pages[0]=emu_ram_ext+(0x10000L-0xa000L)+0x0000L;
    emu_ram_pages[1]=emu_ram_ext+(0x10000L-0xa000L)+0x0000L;
    emu_ram_pages[2]=emu_ram_ext+(0x10000L-0xa000L)+0x0000L;
    emu_ram_pages[3]=emu_ram_ext+(0x10000L-0xa000L)+0x0000L;
  }

  switch (mbc) {
  case 1:
    emu_v_ram_enable =mbc_dummy;
    emu_v_roml_select=mbc1_roml_select;
    emu_v_romh_select=mbc1_roml_select;
    emu_v_ram_select =mbc1_romh_select;
    emu_v_mode_select=mbc1_mode_select;
    mbc1_rom_lsb=0;
    mbc1_rom_msb=0;
    break;
  case 2:
    emu_v_ram_enable =mbc_dummy;
    emu_v_roml_select=mbc2_rom_select;
    emu_v_romh_select=mbc2_rom_select;
    emu_v_ram_select =mbc_dummy;
    emu_v_mode_select=mbc_dummy;
    break;
  case 3:
    emu_v_ram_enable =mbc_dummy;
    emu_v_roml_select=mbc3_rom_select;
    emu_v_romh_select=mbc3_rom_select;
    emu_v_ram_select =mbc3_ram_select;
    emu_v_mode_select=mbc3_rtc_latch;
    break;
  case 5:
    emu_v_ram_enable =mbc_dummy;
    emu_rom_pages[0]=emu_rom-0x4000L;
    emu_v_roml_select=mbc5_rom_select;
    emu_v_romh_select=mbc_dummy;
    emu_v_ram_select =mbc5_ram_select;
    emu_v_mode_select=mbc_dummy;
    break;
  default:
    emu_v_ram_enable =mbc_dummy;
    emu_v_roml_select=mbc_dummy;
    emu_v_romh_select=mbc_dummy;
    emu_v_ram_select =mbc_dummy;
    emu_v_mode_select=mbc_dummy;
    break;
  }

  return TRUE;
}

void cartr_no_info(void) {
  /*--------------------*/
  HEADER  no_info_header={ "","",0,0,0 };

  header=no_info_header;
}

void cartr_info(void) {
  /*-----------------*/
  OBJECT  *o;
  char    *s,f[10];
  int     x,y,w,h,b;

  rsrc_gaddr(R_TREE,DINFO,&o);

  s=o[DIFSIZE].ob_spec.free_string;
  sprintf(f,"%%%ldld",strlen(s));
  sprintf(s,f,header.fsz);

  s=o[DITITLE].ob_spec.free_string;
  sprintf(f,"%%%ld.%lds",strlen(s),strlen(s));
  sprintf(s,f,header.name);

  s=o[DICARTR].ob_spec.free_string;
  sprintf(f,"%%%ld.%lds",strlen(s),strlen(s));
  sprintf(s,f,header.cname);

  s=o[DIROMSZ].ob_spec.free_string;
  sprintf(f,"%%%ldld",strlen(s));
  sprintf(s,f,header.romsz/1024);

  s=o[DIRAMSZ].ob_spec.free_string;
  sprintf(f,"%%%ldld",strlen(s));
  sprintf(s,f,header.ramsz);

  form_center(o,&x,&y,&w,&h);
  form_dial(FMD_START,0,0,0,0,x,y,w,h);
  objc_draw(o,0,99,x,y,w,h);
  b=form_do(o,0);
  objc_change(o,b,0,x,y,w,h,NORMAL,1);
  form_dial(FMD_FINISH,0,0,0,0,x,y,w,h);
}
void program_info(void) {
  /*-----------------*/
  OBJECT  *o;
  int     x,y,w,h,b;

  rsrc_gaddr(R_TREE,DPROINFO,&o);
  form_center(o,&x,&y,&w,&h);
  form_dial(FMD_START,0,0,0,0,x,y,w,h);
  objc_draw(o,0,99,x,y,w,h);
  b=form_do(o,0);
  objc_change(o,b,0,x,y,w,h,NORMAL,1);
  form_dial(FMD_FINISH,0,0,0,0,x,y,w,h);
}

/* === End ====================================================	*/
