/*
 *  K&R Oroogu Runtime Support Package 
 *
 *  Version 0.301
 *
 *  Implemented by Georg Kraml & Angelika Riemer
 * 
 *  Note:
 *
 *       All functions exported by this package assume that they are 
 *       called from C code generated by the K&R Oroogu compiler, and 
 *       that this code is semantically correct. For example, the
 *       functions do not check if the queue and control stack slot
 *       indices handed to them are out of bound.
 *
 *       If the calling code is incorrect due to compiler bugs, the
 *       program may crash.
 *
 *  Further information is available from <http://www.purists.org/oroogu/>.
 *  Please send bug reports to <georg@purists.org>.
 *
 */

#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "orsp.h"

/*
 *  Constants
 */
#define BUFSIZE 20
#define STOP    '~'

/*
 *  Order of numbers stored in a queue item
 */
enum ORDER
{
	ASCEND = 0,
	DESCEND
};

/*
 *  Queue item structure; describes an ordered set of consecutive numbers.
 */
struct qentry
{
	struct qentry *next;
	int lo, hi;
	enum ORDER order;
	const char *string;
};

/*
 *  Queue container structure
 *
 *  This structure encapsulates a linked list of qentry items.
 */
struct queue
{
	struct qentry *begin, **end;
	const char *lexeme;
};

/*
 *  Global variables
 */
const char *cmdname, *string;
int stwidth, stdepth, qnum, *stack;
struct queue *q;

/*
 *  error: report error and abort compilation.
 *
 *  Parameters:
 *
 *    fmt         Format string, xxprintf-style.
 *
 *    ...         Data referred to by the format string.
 */
void
error(const char *fmt, ...)
{
	va_list ap;
	const char *tmp = cmdname;

	while (*tmp) tmp ++;
	while (tmp != cmdname && *(tmp-1) != '/' && *(tmp-1) != '\\') {
		tmp --;
	}
	fprintf(stderr, "%s: ", tmp);
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, ".\n");
	exit(1);	
}

/*
 *  add: add an ordered set of consecutive numbers to a queue.
 *
 *  Parameters:
 *
 *    id          Symbol-table index of destination queue.
 *
 *    start	  First element in the set.
 *
 *    stop        Last element in the set.
 *
 *    string      Optional string to be stored with the set.
 */
void
add(int id, int start, int stop, const char *string)
{
	if ((*q[id].end = malloc(sizeof(struct qentry))) == NULL) {
		error("out of memory while adding items to \"%s\"", 
			q[id].lexeme);
	}
	if (start <= stop) {
		(*q[id].end)->lo = start;
		(*q[id].end)->hi = stop;
		(*q[id].end)->order = ASCEND;
	}
	else {
		(*q[id].end)->lo = stop;
		(*q[id].end)->hi = start;
		(*q[id].end)->order = DESCEND;
	}
	(*q[id].end)->string = string;
	(*q[id].end)->next = NULL;
	q[id].end = &((*q[id].end)->next);
}

/*
 *  copy: add a copy of queue item to another queue.
 *
 *  Parameters:
 *
 *    id          Symbol-table index of destination queue.
 *
 *    p           Pointer to the item to be copied.
 */
void
copy(int id, struct qentry *p)
{
	if ((*q[id].end = malloc(sizeof(struct qentry))) == NULL) {
		error("out of memory while adding items to \"%s\"", 
			q[id].lexeme);
	}
	memcpy(*q[id].end, p, sizeof(struct qentry));
	(*q[id].end)->next = NULL;
	q[id].end = &((*q[id].end)->next);
}

/*
 *  empty: remove all items from a queue
 *
 *  Parameters:
 *
 *    id          Symbol-table index of queue to be emptied
 */
void
empty(int id)
{
	struct qentry *tmp;
	
	while (q[id].begin) {
		tmp = q[id].begin;
		q[id].begin = q[id].begin->next;
		free(tmp);
	}
	q[id].end = &q[id].begin;
}

/*
 *  addall: add all item in a given queue to another queue
 *
 *  Parameters:
 *
 *    src         Symbol-table index of source queue
 *
 *    dest        Index of destination queue
 */
void
addall(int src, int dest)
{
	struct qentry *tmp = q[src].begin;

	while (tmp) {
		copy(dest, tmp);
		tmp = tmp->next;
	}
}

/*
 *  get: remove an item from the queue, throw exception if queue is empty
 *
 *  Parameters:
 *
 *    id          Symbol-table index of queue
 *
 *  Returns:
 *
 *    The value removed from the queue
 */
int
get(int id)
{
	int d;
	struct qentry *tmp = q[id].begin;

	if (tmp == NULL) {
		error("exception: queue \"%s\" is empty", q[id].lexeme);
	}
	if (tmp->string) {
		string = tmp->string;
		if (!(q[id].begin = q[id].begin->next)) {
			q[id].end = &q[id].begin;
		}		
		free(tmp);
		return 1;
	}
	if (tmp->lo != tmp->hi) {
		return (tmp->order == ASCEND) ? tmp->lo++ : tmp->hi--;
	}
	if (!(q[id].begin = q[id].begin->next)) {
		q[id].end = &q[id].begin;
	}
	d = tmp->lo;
	free(tmp);
	return d;	
}

/*
 *  peek: return the queue's front item, throw exception if queue is empty
 *
 *  Parameters:
 *
 *    id          Symbol-table index of queue
 *
 *  Returns:
 *
 *    The queue's front item
 */
int
peek(int id)
{
	struct qentry *tmp = q[id].begin;

	if (tmp == NULL) {
		error("exception: queue \"%s\" is empty", 
			q[id].lexeme);
	}
	if (tmp->string) {
		string = tmp->string;
	}
	return (tmp->order == ASCEND) ? tmp->lo : tmp->hi;
}

/*
 *  peek: throw an exception if a given value is zero.
 *
 *  Parameters:
 *
 *    val         Number to be checked.
 */
int
assert(int val)
{
	if (!val) error("exception: division by zero");
	return val;
}

/*
 *  lval: stores queue indices in control stack slot, empty "$zero" queue.
 *
 *  Parameters:
 *
 *    slot        Index of target slot
 *
 *    ...         List of queue indices, terminated by a value of -1.
 */
void
lval(int slot, ...)
{
	va_list ap;
	int i = slot * stwidth;

	va_start(ap, slot);
	do stack[i] = va_arg(ap, int); while (stack[i++] != -1);
	va_end(ap);
	empty(0);
	string = NULL;
}

/*
 *  range: appends numbers to the "$zero" queue.
 *
 *    begin       First value to be stored
 *
 *    end         Last value to be stored
 *
 *    fill        There are three possible values:
 *
 *      2         The function adds all numbers between "begin" and "end"
 *                to the queue. Besides, it checks if any of the calls to
 *                "get" or "peek" made during calculation of its arguments
 *                could not return a number. If so, an exception is thrown.
 *
 *      1         The function stores just "begin", completely ignoring
 *                "end", but it performs the string check outlined above.
 *
 *      0         The function stores just "begin", completely ignoring
 *                "end"; if a string was found during the calculation
 *                of its parameters this string is stored instead.
 */
void
range(int begin, int end, int fill)
{
	if (fill && string) {
		error("exception: string used in arithmetic expression");
	}
	add(0, begin, (fill == 2) ? end : begin, string);
	string = NULL;
}

/*
 *  addtext: appends string literal to "$zero" queue.
 *
 *    string      String literal to be stored.
 */
void
addtext(const char *string)
{
	add(0, 0, 0, string);
	string = NULL;
}

/*
 *  exp: calculates A to the power of B
 *
 *  Parameters:
 *
 *    b           Base
 *
 *    e           Exponent
 *
 *  Returns;
 *
 *    If "e" is positive, the function returns "b" to the power of "e".
 *    If "e" is zero or negative, zero is returned.
 */
int
exp(int b, int e)
{
	int d = 1;

	if (e < 0) return 0;
	while (e--) {
		d *= b;
	}
	return d;
}

/*
 *  assign: makes control stack queues copies of given source queue.
 *
 *  Parameters:
 *
 *    slot        Element of loop control slot. The queues in this slot
 *                are the function's target queues; i. e, they are going
 *                to be modified by the function.
 *
 *    id          Symbol-table index of source queue
 */
void
assign(int slot, int id)
{
	int i;
	struct qentry *tmpsrc;

	/* if "id" != 0 temporarily store "id" items in "$zero" queue */
	if (id) {
		empty(0);
		addall(id, 0);
	}

	/* delete items from all slot queues */
	i = slot * stwidth;
	while ((id = stack[i++]) != -1) {
		empty(id);
	}

	/* walk through all items in the "$zero" queue */
	tmpsrc = q[0].begin;
	while (tmpsrc) {

		/* walk through all queues in the slot */
		i = slot * stwidth;
		while ((id = stack[i++]) != -1) {
			copy(id, tmpsrc);
		}

		/* advance */
		tmpsrc = tmpsrc->next;
	}
}

/*
 *  append: append source queue items to a set of control stack queues.
 *
 *  Parameters:
 *
 *    slot        Element of loop control stack. The items in the source
 *                queue will be added to the queues whose symbol-table
 *                indices are stored in this array.
 *
 *    id          Symbol-table index of source queue.
 */
void
append(int slot, int id)
{
	int i;
	struct qentry *tmpsrc;

	/* if "id" != 0 temporarily store "id" items in "$zero" queue */
	if (id) {
		empty(0);
		addall(id, 0);
	}

	/* walk through all items in the "$zero" queue */
	tmpsrc = q[0].begin;
	while (tmpsrc) {

		/* walk through all queues in the slot */
		i = slot * stwidth;
		while ((id = stack[i++]) != -1) {
			copy(id, tmpsrc);
		}

		/* advance */
		tmpsrc = tmpsrc->next;
	}
}
  
/*
 *  diffkill: helper function for "diffmatch": remove an item from a queue.
 *
 *  Parameters:
 *
 *    p           Address of pointer to the item to be removed
 */
void
diffkill(struct qentry **p)
{
	struct qentry *tmp = *p;

	*p = (*p)->next;
	free(tmp);
}

/*
 *  diffins: helper function for "diffmatch": insert an item into a queue.
 *
 *  Parameters:
 *
 *    p           Pointer to the new item's predecessor.
 *
 *    lo          Lowest number in the set described by the new item
 *
 *    hi          Highest number in the set
 */
void
diffins(struct qentry *p, int lo, int hi)
{
	struct qentry *tmp = p->next;

	if ((p->next = malloc(sizeof(struct qentry))) == NULL) {
		error("out of memory");
	}
	p->next->lo = lo;
	p->next->hi = hi;
	p->next->order = p->order;
	p->next->string = NULL;
	p->next->next = tmp;
}

/*
 *  diffmatch: helper function for "diff": compare two sets of numbers.
 *
 *  Parameters:
 *
 *    src         Pointer to a set of numbers to be removed.
 *
 *    dest        Address of a pointer to the set from which the numbers
 *                in "src" are to be removed.
 *
 *  Return values:
 *
 *    2           The function found that "src" was not a subset of
 *                "dest", maybe the two sets were even disjoint.
 *                Accordingly, zero or more numbers have been removed
 *                from "dest". However, "src" still contains numbers
 *                that need to be removed from "dest"'s queue,
 *                continue traversal.
 *
 *    1           Same as above, but "dest" was even emptied completely
 *                and has been purged from its queue.
 *
 *    0           The function found that "src" was a subset of "dest".
 *                All the numbers in "src" have been removed from "dest";
 *                stop traversing the queue that contained "dest".
 */
int
diffmatch(struct qentry *src, struct qentry **dest)
{
	int mark;

	if (src->string) {
		if ((*dest)->string 
			&& !strcmp(src->string, (*dest)->string)) {
			diffkill(dest);
			return 0;
		}
		return 2;
	}
	if (src->lo == (*dest)->lo) {
		if (src->hi == (*dest)->hi) {
			diffkill(dest);
			return 0;
		}
		else if (src->hi < (*dest)->hi) {
			(*dest)->lo = src->hi + 1;
			return 0;
		}
		else /* if (src->hi > (*dest)->hi) */ {
			src->lo = (*dest)->hi + 1;
			diffkill(dest);
			return 1;
		}
	}
	else if (src->lo < (*dest)->lo && src->hi >= (*dest)->lo) {
		if (src->hi == (*dest)->hi) {
			diffins(src, (*dest)->hi + 1, src->hi);
			src->hi = (*dest)->lo - 1;
			diffkill(dest);
			return 1;
		}
		else if (src->hi < (*dest)->hi) {
			mark = src->hi;
			src->hi = (*dest)->lo - 1;
			(*dest)->lo = mark + 1;
			return 2;
		}
		else /* if (src->hi > (*dest)->hi) */ {
			src->hi = (*dest)->lo - 1;
			diffkill(dest);
			return 1;
		}
	}
	else if (src->lo > (*dest)->lo && src->lo <= (*dest)->hi) {
		if (src->hi == (*dest)->hi) {
			(*dest)->hi = src->lo - 1;
			return 0;
		}
		else if (src->hi < (*dest)->hi) {
			diffins(*dest, src->hi + 1, (*dest)->hi);
			(*dest)->hi = src->lo - 1;
			return 0;
		}
		else /* if (src->hi > (*dest)->hi) */ {
			mark = (*dest)->hi;
			(*dest)->hi = src->lo - 1;
			src->lo = mark + 1;
			return 2;
		}
	}
	return 2;
}

/*
 *  diff: remove source queue items from a set of control stack queues.
 *
 *  Parameters:
 *
 *    slot        
 *
 *    id          Symbol-table index of source queue.
 */
void
diff(int slot, int id)
{
	int i, cont;
	struct qentry *tmpsrc, **tmpdest;

	/* if "id" != 0 temporarily store "id" items in "$zero" queue */
	if (id) {
		empty(0);
		addall(id, 0);
	}
		
	/* walk through all items in the "$zero" queue */
	tmpsrc = q[0].begin;
	while (tmpsrc) {

		/* walk through all queues in the slot */
		i = slot * stwidth;
		while ((id = stack[i++]) != -1) {
			tmpdest = & q[id].begin;
			while (*tmpdest) {
				cont = diffmatch(tmpsrc, tmpdest);
				if (cont == 2) {
					tmpdest = & (*tmpdest)->next;
				}
				else if (cont == 0) {
					break;
				}
			}
			if (q[id].begin == NULL) {
				q[id].end = & q[id].begin;
			}
		}

		/* advance */
		tmpsrc = tmpsrc->next;
	}
}

/*
 *  loop: tests if at least one queue in a given slot contains elements
 *
 *  Parameters:
 *
 *    slot        Loop control stack element whose loops are to be tested.
 *
 *  Returns:
 *
 *    If at least one queue in the slot contains at least one element, 
 *    the function returns nonzero. If all queues in the slot are
 *    empty, the function returns zero.
 */
int
loop(int slot)
{
	int i = slot * stwidth, id;

	while ((id = stack[i++]) != -1) {
		if (q[id].begin) return 1;
	}
	return 0;
}

/*
 *  startup: set global variables, allocate queues and control structures.
 *
 *  Parameters:
 *
 *    depth       Maximum depth of loop control stack; i. e, one plus
 *                the maximum number of loops surrounding any statement
 *                in the original Oroogu program.
 *
 *    width       One plus the maximum width of loop control stack; i. e,
 *                the maximum number of queues used on the left-hand side 
 *                of a storage operator anywhere in the program.
 *
 *    num         One plus the number of queues used.
 *
 *    cmd         Path to binary being executed, taken from command line.
 */
void
startup(int depth, int width, int num, const char *cmd)
{
	int i;

	stdepth = depth;
	stwidth = width;
	qnum = num;
	cmdname = cmd;
	if ((stack = malloc(stwidth * stdepth * sizeof(int))) == NULL
		|| (q = malloc(qnum * sizeof(struct queue))) == NULL) {
		error("out of memory during initialization");
	}
	for (i = 0; i < qnum; i ++) {
		q[i].begin = NULL;
		q[i].end = &q[i].begin;
	}
	lexeme(0, "$zero");
}

/*
 *  Read string delimited by quotation marks and add it to "input".
 *
 *  Parameters:
 *
 *    id          Symbol-table index of "input "queue.
 */
void
readglued(int id)
{
	int i = 0, bufsize = BUFSIZE, c;
	char *inbuf;

  	if ((inbuf = malloc(BUFSIZE)) == NULL) {
		error("out of memory");
	}
	while ((c = getchar()) != EOF) {
		if (c == '"') {
			break;
		}
		if (i == bufsize - 1) {
			bufsize *= 2;
			inbuf = realloc(inbuf, bufsize);
			if (inbuf == NULL) {
				error("out of memory");
			}
		}
		inbuf[i++] = c;
	}
	inbuf[i] = '\0';
	add(id, 0, 0, inbuf);
}

/*
 *  Set up "input" queue
 *
 *  Parameters:
 *
 *    id          Symbol-table index of "input" queue.
 */
void
readinput(int id)
{
	int c, val;
	char *inbuf;

	printf("Reading input from stdin, hit ^D to continue:\n");
	while ((c = getchar()) != EOF) {
		if (c == '\r' && (c = getchar()) != '\n') {
			if (ungetc(c, stdin) == EOF) {
				error("cannot unread character");
			}
			add(id, 0, 0, "\r");
		}
		if (c == '\n') {
			add(id, 0, 0, "\n");
		}
		else if (isdigit(c)) {
			if (ungetc(c, stdin) == EOF) {
				error("cannot unread character");	
			}
			if (scanf("%d", &val) == 0) {
				error("cannot read from stdin");
			}
			add(id, val, val, NULL);
		}
		else if (c == '"') {
			readglued(id);
		}
		else if (!isspace(c)) {
			if ((inbuf = malloc(2) ) == NULL) {
				error("out of memory");
			}
			inbuf[0] = c;
			inbuf[1] = '\0';
			add(id, 0, 0, inbuf);
		}
	}
	printf("Processing...\n");
}

/*
 *  lexeme: set the lexeme associated with a given queue.
 *
 *  Arguments:
 *
 *    id          Symbol-table index of queue.
 *
 *    lex         Lexeme associated with queue.
 */
void
lexeme(int id, const char *lex)
{
	q[id].lexeme = lex;
	if (!strcmp(lex, "input")) {
		readinput(id);
	}
}

/*
 *  dumpitem: helper function for "cleanup": dump a queue item.
 *
 *  Parameters:
 *
 *    p           Pointer to item to be dumped.
 *
 *    dump        Output stream to be dumped to.
 */
void
dumpitem(struct qentry *p, FILE *dump)
{
	char *sep;

	if (p->string) {
		fprintf(dump, "%s", p->string);
	}
	else if (p->lo == p->hi) {
		fprintf(dump, "%02d ", p->hi);
	}
	else if (p->order == ASCEND) {
		sep = (p->lo < p->hi - 1) ? ".." : " ";
		fprintf(dump, "%02d%s%02d ", p->lo, sep, p->hi);
	}
	else {
		sep = (p->hi > p->lo + 1) ? ".." : " ";
		fprintf(dump, "%02d%s%02d ", p->hi, sep, p->lo);
	}
}

/*
 *  cleanup: dump variables
 */
void
cleanup()
{
	int i;
	FILE *dump;
	struct qentry *tmp;

	for (i = 1; i < qnum; i ++) {
		switch (q[i].lexeme[0]) {
		case 'd':
			dump = stdout;
			break;

		case 'e':
			dump = stderr;
			break;

		default:
			dump = NULL;
		} 
		if (dump && q[i].begin) {
			fprintf(dump, "%s: ", q[i].lexeme);
			while (q[i].begin) {
				tmp = q[i].begin;
				q[i].begin = q[i].begin->next;
				dumpitem(tmp, dump);
				free(tmp);
			}
			fprintf(dump, "\n");
		}
	}
}
